import os
import shutil
import yaml
from false_presuppositions import NumericalFalsePresuppositionEvaluator
from scientific_attribution import ReferencesEvaluator
from historical_events import InterpersonalRelationshipEvaluator
from openai import OpenAI
import pandas as pd 
import argparse
from tqdm import tqdm


def read_api_keys(config_file="config.yml"):
    with open(config_file, 'r') as file:
        config = yaml.safe_load(file)
    return config['openai_api_key'], config['together_api_key'], config['s2_api_key']


def get_args():
    parser = argparse.ArgumentParser(
        description="Process files in a folder using various evaluators.")


    parser.add_argument("--api_key_file", default="config.yml",
                        help="Path to the text file containing API keys.")

    parser.add_argument(
        "--input_dir", help="Path to the folder where output files will be saved.", default="./")

    parser.add_argument(
        "--output_dir", help="Path to the folder where output files will be saved.", default="./hlc")

    parser.add_argument("--model", help="Model to use from Together AI API for verifiers. Defaults to 'meta-llama/Llama-3.3-70B-Instruct-Turbo'.",
                        default="meta-llama/Llama-3.3-70B-Instruct-Turbo")

    parser.add_argument("--false_presupposition", action="store_true",
                        help="Process false presupposition evaluator.")
    parser.add_argument("--scientific_attribution", action="store_true",
                        help="Process scientific attribution evaluator.")
    parser.add_argument("--historical_events", action="store_true",
                        help="Process historical events evaluator.")

    args = parser.parse_args()
    return args


if __name__ == "__main__":
    args = get_args()

    openai_api_key, together_api_key, s2_api_key = read_api_keys(
        args.api_key_file)

    client = OpenAI(api_key = openai_api_key)
    data = pd.read_csv("prompts_falsepresuppositions.csv")
    data_he = pd.read_csv("prompts_historicalevents.csv")


    if args.false_presupposition:
        input_dir = args.input_dir  
        for filename in tqdm(os.listdir(input_dir)):
            file_path = os.path.join(input_dir, filename)
            nfp_evaluator = NumericalFalsePresuppositionEvaluator(client)
            nfp_evaluator.evaluate(filename, file_path, args.output_dir, data=data)

    if args.scientific_attribution:
        input_dir = args.input_dir  
        for filename in tqdm(os.listdir(input_dir)):
            file_path = os.path.join(input_dir, filename)
            scientific_evaluator = ReferencesEvaluator(s2_api_key, client)
            scientific_evaluator.evaluate_references(
                filename, file_path, args.output_dir)
            
    if args.historical_events:
        input_dir = args.input_dir  
        for filename in tqdm(os.listdir(input_dir)):
            file_path = os.path.join(input_dir, filename)
            he_evaluator = InterpersonalRelationshipEvaluator(client)
            he_evaluator.evaluate(
                filename, file_path, args.output_dir, data=data_he)